/*
* Creation date : Fri Mar 02 14:15:51 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of 
* CE2_DES() function. 
*
* \version CE2_DES.c#1:csrc:1
* \author Pavel Sasunkevich
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "CE2_DES.h"
#include "LLF_DES.h"

/************************ Defines *****************************/
/************************ Enums *******************************/
/************************ Typedefs ****************************/
/************************ Global Data *************************/
/************************ Private function prototype **********/
/************************ Private Functions *******************/
/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
*  CE2_DES
*
* Inputs:
* @param IV_ptr [in\out] - This parameter is the buffer of the IV;
* @param Key_ptr [in] - A pointer to the user's key buffer;
* @param NumOfKeys [in] - The number of keys used: 1, 2, or 3 (defined in the enum);
* @param EncryptDecryptFlag [in] - A flag that determines if the DES should perform;
* @param OperationMode [in] - The operation mode: ECB or CBC;
* @param DataIn_ptr [in] - The pointer to the buffer of the input data to the DES;
* @param DataInSize [in] - The size of the input data.
* @param DataOut_ptr [out] - The pointer to the buffer of the output data from the DES;
*
* Outputs: @returns \b
*  CE2Error_t  
*  - CE2_OK - On success
*  - Otherwise - error code:
*   - CE2_DES_INVALID_KEY_PTR_ERROR
*   - CE2_DES_ILLEGAL_NUM_OF_KEYS_ERROR
*   - CE2_DES_ILLEGAL_ENCRYPT_DECRYPT_FLAG_ERROR
*   - CE2_DES_ILLEGAL_HASH_OP_MODE_ERROR
*   - CE2_DES_INVALID_DATA_IN_PTR_ERROR
*   - CE2_DES_INVALID_DATA_IN_SIZE_ERROR
*
* \brief \b
* Description:
*   This function is used to operate the DES machine in one integrated
* operation: an Encrypt or a Decrypt operation, using operation
* mode: ECB or CBC.
*
* \b
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function LLF_DES 
*     to operate the DES machine with using LibTomCrypt.
***************************************************************/
CE2CIMPORT_C CE2Error_t CE2_DES  (  CE2_DES_Iv_t            IV_ptr,			/* in,out */
                                 CE2_DES_Key_t           *Key_ptr,			/* in */
                                 CE2_DES_NumOfKeys_t     NumOfKeys,			/* in */
                                 CE2_DES_EncryptMode_t   EncryptDecryptFlag,/* in */
                                 CE2_DES_OperationMode_t OperationMode,		/* in */
                                 DxUint8_t               *DataIn_ptr,		/* in */ 
                                 DxUint32_t              DataInSize,		/* in */ 
                                 DxUint8_t               *DataOut_ptr )		/* out */
{
	if (Key_ptr == DX_NULL)
		return CE2_DES_INVALID_KEY_PTR_ERROR;

	if (NumOfKeys < CE2_DES_1_KeyInUse || NumOfKeys >= CE2_DES_NumOfKeysOptions)
		return CE2_DES_ILLEGAL_NUM_OF_KEYS_ERROR;

	if (EncryptDecryptFlag < 0 || EncryptDecryptFlag >= CE2_DES_EncryptNumOfOptions)
		return CE2_DES_ILLEGAL_ENCRYPT_DECRYPT_FLAG_ERROR;

	if (OperationMode < 0 || OperationMode >= CE2_DES_NumOfModes)
		return CE2_DES_ILLEGAL_HASH_OP_MODE_ERROR;

	if (DataIn_ptr == DX_NULL)
		return CE2_DES_INVALID_DATA_IN_PTR_ERROR;

	if (DataInSize == 0 || (DataInSize % 8) != 0)
		return CE2_DES_INVALID_DATA_IN_SIZE_ERROR;

	return LLF_DES(IV_ptr, Key_ptr, NumOfKeys, EncryptDecryptFlag,
					OperationMode, DataIn_ptr, DataInSize, DataOut_ptr);
} /* End of CE2_DES */
